/*
 * DamageObject.h
 *
 * Created 8/17/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/17/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef DAMAGE_OBJECT_H
 #define DAMAGE_OBJECT_H
 
 template <typename T> class DamageObject;
 
 #include "CollisionObject.h"
 
 // For collision handling
 #include "CollisionHandlerEventNodeCollection.h"
 
 #include "global.h"
 
 // For applying effects
 #include "EffectInfo.h"
 #include "EffectInfoCollection.h"
 
 /**
  * Class specification for DamageObject
  * DamageObjects are Objects that deal damage
  */
 template <typename T>
 class DamageObject : virtual public CollisionObject<T>
 {
 // Data Members
 private:
    T _damage;
    EffectInfoCollection<T> _effect_info_collection;
 
 // Local Functions
 public:
    DamageObject( const std::string& name, const T& damage = ZERO, const EffectInfoCollection<T>& effect_info_collection = EffectInfoCollection<T>() ); 
    DamageObject( const DamageObject<T>& dmg_obj );
    virtual ~DamageObject();
    inline DamageObject<T>& operator=( const DamageObject<T>& dmg_obj );
    virtual inline void add_collider( CollisionHandlerEventNode<T>* collider_Ptr );
    virtual inline void add_collider( const std::string& key, CollisionHandlerEventNode<T>* collider_Ptr );
    virtual inline void handle_from_collision( Object<T>& into, const CollisionEntry& c_entry );
    virtual inline void handle_into_collision( Object<T>& from, const CollisionEntry& c_entry );
    virtual inline bool is_damage_object() const;
    virtual inline void load_model( WindowFramework& window );
    virtual inline void load_model( WindowFramework& window, const NodePath& parent );
    inline T get_damage() const;
    inline void set_damage( const T& damage );
    inline void add_effect( EffectInfo<T>* effect_info_Ptr );
    
    virtual void reparent_to(Object<T>& other, double time_action_will_begin = 0.0, int sort = 0,
                   Thread* current_thread = Thread::get_current_thread()) { NodePath::reparent_to( other, sort, current_thread ); }
    virtual void wrt_reparent_to(Object<T>& other, double time_action_will_begin = 0.0, int sort = 0,
                   Thread* current_thread = Thread::get_current_thread()) { NodePath::wrt_reparent_to( other, sort, current_thread ); }
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "DamageObject<" + template_type + ">" );
                            }*/
 
 // Private Functions
 private:
    inline void apply_effects( Object<T>& into, const CollisionEntry& c_entry );
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 DamageObject<T>::DamageObject( const std::string& name, const T& damage, 
                                const EffectInfoCollection<T>& effect_info_collection )
                 : CollisionObject<T>(),
                   _damage( damage ),
                   _effect_info_collection( effect_info_collection )
 {
    NodePath::operator=( new PandaNode( name ) );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 DamageObject<T>::DamageObject( const DamageObject<T>& dmg_obj )
                 : CollisionObject<T>( static_cast< CollisionObject<T> >( dmg_obj ) ),
                   _damage( dmg_obj.damage ),
                   _effect_info_collection( dmg_obj._effect_info_collection )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 DamageObject<T>::~DamageObject()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified DamageObject to this DamageObject.
  *
  * @param (const DamageObject<T>& dmg_obj )
  * @return DamageObject<T>&
  */
 template <typename T>
 inline DamageObject<T>& DamageObject<T>::operator=( const DamageObject<T>& dmg_obj )
 {
    CollisionObject<T>::operator=( static_cast< CollisionObject<T> >( dmg_obj ) );
    _damage = dmg_obj._damage;
    _effect_info_collection = dmg_obj._effect_info_collection;
    
    return *this;
 }
 
 /**
  * add_collider() adds the Collider, specified by the collider_Ptr, using its name as the key, 
  * into the ColliderCollection for this Object.
  *
  * @param (CollisionHandlerEventNode<T>*) collider_Ptr
  */
 template <typename T>
 inline void DamageObject<T>::add_collider( CollisionHandlerEventNode<T>* collider_Ptr )
 {
    nassertv( collider_Ptr != NULL );
    nassertv( !collider_Ptr->is_empty() );
    DamageObject<T>::add_collider( collider_Ptr->get_name(), collider_Ptr );
 }
 
 /**
  * add_collider() adds the Collider specified by the collider_Ptr with the specified key into the
  * ColliderCollection for this Object.
  *
  * @param (const std::string&) key
  * @param (CollisionHandlerEventNode<T>*) collider_Ptr
  */
 template <typename T>
 inline void DamageObject<T>::add_collider( const std::string& key, CollisionHandlerEventNode<T>* collider_Ptr )
 {
    nassertv( collider_Ptr != NULL );
    CollisionObject<T>::add_collider_without_tagging( key, collider_Ptr );
    
    // Tag the Collider as belonging to the Object this DamageObject belongs to
    if ( NodePath::has_tag(OBJECT_KEY_TAG) )
    {
        collider_Ptr->set_tag( OBJECT_KEY_TAG, NodePath::get_tag(OBJECT_KEY_TAG) );
    }
    else // this DamageObject does not have an object tag, then it is independent 
         // (thus, tag its collider Object Key as the key of the DamageObject itself)
    {
        collider_Ptr->set_tag( OBJECT_KEY_TAG, StringConversion::to_str( NodePath::get_key() ) );
    }
    
    // Tag the Collider as belonging to this DamageObject (and also mark the Collider as belonging to a DamageObject)
        collider_Ptr->set_tag( DAMAGE_OBJECT_KEY_TAG, NodePath::get_name() );//StringConversion::to_str( NodePath::get_key() ) );
 }
 
 /**
  * handle_from_collision() handles the collision assuming this Object has
  * collided into another Object (additional information can be found 
  * inside of the CollisionEntry). That is, this Object is assumed to be
  * the "from" Object.
  *
  * @param (Object<T>&) into - the "into" Object
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void DamageObject<T>::handle_from_collision( Object<T>& into, const CollisionEntry& c_entry )
 {
    //NodePath::detach_node();
    
    if ( into.is_prone_to_effects() )
        DamageObject<T>::apply_effects( into, c_entry );
 }
 
 /**
  * handle_into_collision() handles the collision assuming another Object
  * has collided into this Object (additional information can be found 
  * inside of the CollisionEntry). That is, this Object is assumed to be
  * the "into" Object.
  *
  * @param (Object<T>&) from - the "from" Object
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void DamageObject<T>::handle_into_collision( Object<T>& from, const CollisionEntry& c_entry )
 {
    //NodePath::detach_node();
    
    if ( from.is_prone_to_effects() )
        DamageObject<T>::apply_effects( from, c_entry );
 }
 
 /**
  * is_damage_object() returns true if this is a DamageObject; otherwise,
  * false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool DamageObject<T>::is_damage_object() const
 {
    return true;
 }
 
 /**
  * load_model() loads the Object model for the specified WindowFramework. For the
  * Object to be visible in the world, it needs to be reparented to a NodePath
  * that is part of the specified WindowFramework's scene graph.
  *
  * @param (WindowFramework&) window
  */
 template <typename T> 
 inline void DamageObject<T>::load_model( WindowFramework& window )
 {
    // there are no models to load
    // NodePath::operator=( new PandaNode( "DamageObject" ) );
 }
 
 /**
  * load_model() loads the Object model into the specified WindowFramework under
  * the specified parent in the scene graph of the specified WindowFramework.
  *
  * @param (WindowFramework&) window
  * @param (const NodePath&) parent
  */
 template <typename T> 
 inline void DamageObject<T>::load_model( WindowFramework& window, const NodePath& parent )
 {
    DamageObject<T>::load_model( window );
 
    // For the object to be visible, the NodePath must be reparented to the scene graph
    // Here, the NodePath is reparented to the specified parent
    NodePath::reparent_to( parent );
 }
 
 /**
  * get_damage() returns the damage value for this DamageObject.
  *
  * @return T
  */
 template <typename T>
 inline T DamageObject<T>::get_damage() const
 {
    return _damage;
 }
 
 /**
  * set_damage() sets the damage value for this DamageObject to the specified damage value.
  *
  * @param (const T&) damage
  */
 template <typename T>
 inline void DamageObject<T>::set_damage( const T& damage )
 {
    _damage = damage;
 }
 
 /**
  * add_effect() adds the specified EffectInfo pointer to this DamageObject.
  *
  * @param (EffectInfo<T>* const) effect_info_Ptr
  */
 template <typename T>
 inline void DamageObject<T>::add_effect( EffectInfo<T>* effect_info_Ptr )
 {
    _effect_info_collection.add( effect_info_Ptr );
 }
 
 /** PRIVATE FUNCTIONS **/
 
 /**
  * apply_effects() applies all of the effects of this DamageObject onto the
  * specified Object.
  *
  * @param (Object<T>&) object
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void DamageObject<T>::apply_effects( Object<T>& object, const CollisionEntry& c_entry )
 {
    EffectInfoCollection<T>::iterator effect_info_Itr( _effect_info_collection.begin() );
    EffectInfoCollection<T>::iterator end_Itr( _effect_info_collection.end() );
    while ( effect_info_Itr != end_Itr )
    {
        effect_info_Itr->apply_effect( object, c_entry );
        ++effect_info_Itr;
    }
 }
 
 #endif // DAMAGE_OBJECT_H